
/*
*
*       Class   Fmath
*
*       USAGE:  Mathematical class that supplements java.lang.Math and contains:
*           the main physical constants
*           trigonemetric functions ansent from java.lang.Math
*           some useful additional mathematicl functions
*           some conversion functions
*
*       WRITTEN BY: Mick Flanagan
*
*       DATE:    June 2002
*       AMENDED: 7 May 2003
*
*       DOCUMENTATION:
*       See Michael T Flanagan's JAVA library on-line web page:
*       Fmath.html
*
*
**********************************************************/

public class Fmath{

        // PHYSICAL CONSTANTS

        public static final double N_AVAGADRO = 6.0221419947e23;        /*      mol^-1          */
        public static final double K_BOLTZMANN = 1.380650324e-23;       /*      J K^-1          */
        public static final double H_PLANCK = 6.6260687652e-34;         /*      J s             */
        public static final double H_PLANCK_RED = H_PLANCK/(2*Math.PI); /*      J s             */
        public static final double C_LIGHT = 2.99792458e8;              /*      m s^-1          */
        public static final double R_GAS = 8.31447215;                  /*      K^-1 mol^-1     */
        public static final double F_FARADAY = 9.6485341539e4;          /*      C mol^-1        */
        public static final double T_ABS = -273.15;                     /*      Celsius         */
        public static final double Q_ELECTRON = -1.60217646263e-19;     /*      C               */
        public static final double M_ELECTRON = 9.1093818872e-31;       /*      kg              */
        public static final double M_PROTON = 1.6726215813e-27;         /*      kg              */
        public static final double M_NEUTRON = 1.6749271613e-27;        /*      kg              */
        public static final double EPSILON_0 = 8.854187817e-12;         /*      F m^-1          */
        public static final double MU_0 = Math.PI*4e-7;                 /*      H m^-1 (N A^-2) */


        // METHODS
        // Log to base 10 of a double number
        public static double log10(double a){
                return Math.log(a)/Math.log(10.0D);
        }

        // Log to base 10 of a float number
        public static float log10(float a){
                double aa = (double) a;
                return (float) (Math.log(aa)/Math.log(10.0D));
        }

        // Square of a double number
        public static double square(double a){
                return a*a;
        }

        // Square of a float number
        public static float square(float a){
                return a*a;
        }

        // Square of an int number
        public static int square(int a){
                return a*a;
        }

        // factorial of n
        // argument and return are integer, therefore limited to 0<=n<=12
        // see below for long and double arguments
        public static int factorial(int n){
                if(n<0)throw new IllegalArgumentException("n must be a positive integer");
                if(n>12)throw new IllegalArgumentException("n must less than 13 to avoid integer overflow\nTry long or double argument");
                int f = 1;
                for(int i=1; i<=n; i++)f*=i;
                return f;
        }

        // factorial of n
        // argument and return are long, therefore limited to 0<=n<=20
        // see below for double argument
        public static long factorial(long n){
                if(n<0)throw new IllegalArgumentException("n must be a positive integer");
                if(n>20)throw new IllegalArgumentException("n must less than 21 to avoid long integer overflow\nTry double argument");
                long f = 1;
                for(int i=1; i<=n; i++)f*=i;
                return f;
        }

        // factorial of n
        // Argument is of type double but must be, numerically, an integer
        // factorial returned as double but is, numerically, should be an integer
        // numerical rounding may makes this an approximation after n = 21
        public static double factorial(double n){
                if(n<0 || (n-(int)n)!=0)throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
                double f = 1.0D;
                int nn = (int)n;
                for(int i=1; i<=nn; i++)f*=i;
                return f;
        }

        // log to base e of the factorial of n
        // log[e](factorial) returned as double
        // numerical rounding may makes this an approximation
        public static double logFactorial(int n){
                if(n<0 || (n-(int)n)!=0)throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
                double f = 0.0D;
                for(int i=2; i<=n; i++)f+=Math.log(i);
                return f;
        }

        // log to base e of the factorial of n
        // Argument is of type double but must be, numerically, an integer
        // log[e](factorial) returned as double
        // numerical rounding may makes this an approximation
        public static double logFactorial(double n){
        if(n<0 || (n-(int)n)!=0)throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
                double f = 0.0D;
                int nn = (int)n;
                for(int i=2; i<=nn; i++)f+=Math.log(i);
                return f;
        }

        // Returns the length of the hypotenuse of a and b
        // i.e. sqrt(a*a+b*b) [without unecessary overflow or underflow]
        // double version
        public static double hypot(double aa, double bb){
                double amod=Math.abs(aa);
                double bmod=Math.abs(bb);
                double cc = 0.0D, ratio = 0.0D;
                if(amod==0.0){
                        cc=bmod;
                }
                else{
                        if(bmod==0.0){
                                cc=amod;
                        }
                        else{
                                if(amod>=bmod){
                                        ratio=bmod/amod;
                                        cc=amod*Math.sqrt(1.0 + ratio*ratio);
                                }
                                else{
                                        ratio=amod/bmod;
                                        cc=bmod*Math.sqrt(1.0 + ratio*ratio);
                                }
                        }
                }
                return cc;
        }

        // Returns the length of the hypotenuse of a and b
        // i.e. sqrt(a*a+b*b) [without unecessary overflow or underflow]
        // float version
        public static float hypot(float aa, float bb){
                return (float) hypot((double) aa, (double) bb);
        }


        // Maximum of a 1D array of doubles, aa
        public static double maximum(double[] aa){
                int i = 0, n = aa.length;
                double aamax=aa[0];
                for(i=1; i<n; i++){
                        if(aa[i]>aamax)aamax=aa[i];
                }
                return aamax;
        }

        // Maximum of a 1D array of floats, aa
        public static float maximum(float[] aa){
                int i = 0, n = aa.length;
                float aamax=aa[0];
                for(i=1; i<n; i++){
                        if(aa[i]>aamax)aamax=aa[i];
                }
                return aamax;
        }

        // Maximum of a 1D array of ints, aa
        public static int maximum(int[] aa){
                int i = 0, n = aa.length;
                int aamax=aa[0];
                for(i=1; i<n; i++){
                        if(aa[i]>aamax)aamax=aa[i];
                }
                return aamax;
        }

        // Minimum of a 1D array of doubles, aa
        public static double minimum(double[] aa){
                int i = 0, n = aa.length;
                double aamin=aa[0];
                for(i=1; i<n; i++){
                        if(aa[i]<aamin)aamin=aa[i];
                }
                return aamin;
        }

        // Minimum of a 1D array of floats, aa
        public static float minimum(float[] aa){
                int i = 0, n = aa.length;
                float aamin=aa[0];
                for(i=1; i<n; i++){
                        if(aa[i]<aamin)aamin=aa[i];
                }
                return aamin;
        }

        // Minimum of a 1D array of ints, aa
        public static int minimum(int[] aa){
                int i = 0, n = aa.length;
                int aamin=aa[0];
                for(i=1; i<n; i++){
                        if(aa[i]<aamin)aamin=aa[i];
                }
                return aamin;
        }


        /*      returns -1 if x < 0 else returns 1   */
        //  double version
        public static double sign(double x){
                if (x<0.0){
                        return -1.0;
                }
                else{
                        return 1.0;
                }
        }

        /*      returns -1 if x < 0 else returns 1   */
        //  float version
                public static float sign(float x){
                if (x<0.0F){
                        return -1.0F;
                }
                else{
                        return 1.0F;
                }
        }

        /*      returns -1 if x < 0 else returns 1   */
        //  int version
        public static int sign(int x){
                if (x<0){
                        return -1;
                }
                else{
                        return 1;
                }
        }

        /*      returns -1 if x < 0 else returns 1   */
        // long version
        public static long sign(long x){
                if (x<0){
                        return -1;
                }
                else{
                        return 1;
                }
        }

        // UNIT CONVERSIONS

        // Converts electron volts(eV) to corresponding wavelength in nm
        public static double evToNm(double ev){
                return  1e+9*C_LIGHT/(-ev*Q_ELECTRON/H_PLANCK);
        }

        // Converts wavelength in nm to matching energy in eV
        public static double nmToEv(double nm)
        {
                return  C_LIGHT/(-nm*1e-9)*H_PLANCK/Q_ELECTRON;
        }

        // Converts Celsius to Kelvin
        public static double celsiusToKelvin(double cels){
                return  cels-T_ABS;
        }

        // Converts Kelvin to Celsius
        public static double kelvinToCelsius(double kelv){
                return  kelv+T_ABS;
        }

        // Converts Celsius to Fahrenheit
        public static double celsiusToFahren(double cels){
                return  cels*(9.0/5.0)+32.0;
        }

        // Converts Fahrenheit to Celsius
        public static double fahrenToCelsius(double fahr){
                return  (fahr-32.0)*5.0/9.0;
        }

        // Converts calories to Joules
        public static double calorieToJoule(double cal){
                return  cal*4.1868;
        }

        // Converts Joules to calories
        public static double jouleToCalorie(double joule){
                return  joule*0.23884;
        }

        // ADDITIONAL TRIGONOMETRIC FUNCTIONS
        // Cotangent
        public static double cot(double a){
                return 1.0/Math.tan(a);
        }

        // Secant
        public static double sec(double a){
                return 1.0/Math.cos(a);
        }

        // Cosecant
        public static double csc(double a){
                return 1.0/Math.sin(a);
        }

        // Inverse functions of real numbers

        // Inverse cotangent
        public static double acot(double a){
                return Math.atan(1.0/a);
        }

        public static double acot2(double a, double b){
                return Math.atan2(b, a);
        }

        // Inverse secant
        public static double asec(double a){
                return Math.acos(1.0/a);
        }

        // Inverse cosecant
        public static double acsc(double a){
                return Math.asin(1.0/a);
        }

        //Hyperbolic sine of a double number
        public static double sinh(double a){
                return 0.5*(Math.exp(a)-Math.exp(-a));
        }

        //Hyperbolic cosine of a double number
        public static double cosh(double a){
                return 0.5*(Math.exp(a)+Math.exp(-a));
        }

        //Hyperbolic tangent of a double number
        public static double tanh(double a){
                return sinh(a)/cosh(a);
        }

        //Hyperbolic cotangent of a double number
        public static double coth(double a){
                return 1.0/tanh(a);
        }

        //Hyperbolic secant of a double number
        public static double sech(double a){
                return 1.0/cosh(a);
        }

        //Hyperbolic cosecant of a double number
        public static double csch(double a){
                return 1.0/sinh(a);
        }

        // Inverse hyberbolic functions of real numbers

        // Inverse hyperbolic sine of a double number
        public static double asinh(double a){
                return Math.log(a+Math.sqrt(a*a+1));
        }

        // Inverse hyperbolic cosine of a double number
        public static double acosh(double a){
                return Math.log(a+Math.sqrt(a*a-1));
        }

        // Inverse hyperbolic tangent of a double number
        public static double atanh(double a){
                return 0.5*(Math.log(1.0 + a)-Math.log(1.0 + a));
        }

        // Inverse hyperbolic cotangent of a double number
        public static double acoth(double a){
                return 0.5*(Math.log(1.0 + a)-Math.log(a - 1.0));
        }

        // Inverse hyperbolic secant of a double number
        public static double asech(double a){
                return 0.5*(Math.log(1.0/a + Math.sqrt(1.0/a - 1.0)));
        }

        // Inverse hyperbolic cosecant of a double number
        public static double acsch(double a){
                return 0.5*(Math.log(1.0/a + Math.sqrt(1.0/a + 1.0)));
        }

        // MANTISSA ROUNDING (TRUNCATING)
        // Rounds the mantissa of a double to prec places
        public static double truncate(double x, int prec){

                if(prec<0)return x;
                if(x==0.0D)return x;
                if(Fmath.isNaN(x))return x;
                if(Fmath.isPlusInfinity(x))return x;
                if(Fmath.isMinusInfinity(x))return x;

                if(x==1.0D/0.0D || x==-1.0D/0.0D)return x;
                String ss = "", newss = "";
                Double xx = new Double(x);

                ss = xx.toString();
                newss = ssround(ss, prec);

                return Double.parseDouble(newss);
        }

        // Rounds the mantissa of a float to prec places
        public static float truncate(float x, int prec){

                if(prec<0)return x;
                if(x==0.0F)return x;
                if(Fmath.isNaN(x))return x;
                if(Fmath.isPlusInfinity(x))return x;
                if(Fmath.isMinusInfinity(x))return x;

                String ss = "", newss = "";
                Float xx = new Float(x);

                ss = xx.toString();
                newss = ssround(ss, prec);
                return Float.parseFloat(newss);
        }

        // method for Fmath.truncate
        private static String ssround(String ss, int prec){

                String newss1 = "", newss2 = "", newss3 = "";
                int i = 0, posdot = 0, pose = 0, posf = 0, posplace = 0;

                if(prec<0)throw new IllegalArgumentException("precision less than zero places");

                posf = ss.length()-1;
                pose = ss.indexOf('E');
                posdot = ss.indexOf('.');
                posplace = posdot+prec;

                if(pose>0){
                        if(posplace<pose){
                                if(posplace+1<pose){
                                        newss1 = subround(ss.substring(0,posplace+2));
                                }
                                else{
                                        newss1 = ss.substring(0,posplace+1);
                                }
                        }
                        else{
                                newss1 = ss.substring(0,pose);
                        }
                        newss2 = ss.substring(pose, posf+1);
                        newss3 = newss1.concat(newss2);
                }
                else{
                        if(posplace<posf){
                                if(posplace+1<=posf){
                                        newss3 = subround(ss.substring(0,posplace+2));
                                }
                                else{
                                        newss3 = ss.substring(0,posplace+1);
                                }
                        }
                        else{
                                newss3 = ss;
                        }
                }
                return newss3;
        }

        // method for the ssround method
        private static String subround(String ss){
                int i = 0, j = 0, k = 0, posdot = 0;
                int n = ss.length();
                int[] iss = new int[n+1];
                boolean finish = false;
                String newss = "";

                posdot=ss.indexOf('.');
                for(i=0; i<n; i++){
                        iss[i] = (int)ss.charAt(i);
                }
                if(iss[n-1]<52){
                        finish=true;
                }

                i=n-2;
                while(!finish){
                        j=i;
                        if(j==posdot)j--;
                        iss[j]++;
                        if(iss[j]<58){
                                finish=true;
                        }
                        else{
                                if(j==posdot){
                                        for(k=n-1; k>=posdot; k--){
                                                iss[k+1]=iss[k];
                                        }
                                        iss[posdot]=48;
                                        iss[posdot-1]=49;
                                        n++;
                                        finish=true;
                                }
                                else{
                                        iss[j]=48;
                                }
                        }
                        i--;
                }
                StringBuffer strbuff = new StringBuffer();
                for(k=0; k<n-1; k++){
                        strbuff.append((char)iss[k]);
                }
                newss=strbuff.toString();
                return newss;
        }

        // Returns true if x is plus infinity
        // x is double
        public static boolean isPlusInfinity(double x){
                boolean test=false;
                if(x==Double.POSITIVE_INFINITY)test=true;
                return test;
        }

        // Returns true if x is plus infinity
        // x is float
        public static boolean isPlusInfinity(float x){
                boolean test=false;
                if(x==Float.POSITIVE_INFINITY)test=true;
                return test;
        }

        // Returns true if x is minus infinity
        // x is double
        public static boolean isMinusInfinity(double x){
        boolean test=false;
                if(x==Double.NEGATIVE_INFINITY)test=true;
                return test;
        }

        // Returns true if x is minus infinity
        // x is float
        public static boolean isMinusInfinity(float x){
                boolean test=false;
                if(x==Float.NEGATIVE_INFINITY)test=true;
                return test;
        }


        // Returns true if x is 'Not a Number' (NaN)
        // x is double
        public static boolean isNaN(double x){
                boolean test=false;
                if(x!=x)test=true;
                return test;
        }

        // Returns true if x is 'Not a Number' (NaN)
        // x is float
        public static boolean isNaN(float x){
                boolean test=false;
                if(x!=x)test=true;
                return test;
        }

        // Returns true if x equals y
        // x and y are double
        // x may be float within range, PLUS_INFINITY, NEGATIVE_INFINITY, or NaN
        // NB!! This method treats two NaNs as equal
        public static boolean isEqual(double x, double y){
                boolean test=false;
                if(Fmath.isNaN(x)){
                        if(Fmath.isNaN(y))test=true;
                }
                else{
                        if(Fmath.isPlusInfinity(x)){
                                if(Fmath.isPlusInfinity(y))test=true;

                        }
                        else{
                                if(Fmath.isMinusInfinity(x)){
                                        if(Fmath.isMinusInfinity(y))test=true;
                                }
                                else{
                                        if(x==y)test=true;
                                }
                        }
                }
                return test;
        }

        // Returns true if x equals y
        // x and y are float
        // x may be float within range, PLUS_INFINITY, NEGATIVE_INFINITY, or NaN
        // NB!! This method treats two NaNs as equal
        public static boolean isEqual(float x, float y){
                boolean test=false;
                if(Fmath.isNaN(x)){
                        if(Fmath.isNaN(y))test=true;
                }
                else{
                        if(Fmath.isPlusInfinity(x)){
                                if(Fmath.isPlusInfinity(y))test=true;
                        }
                        else{
                                if(Fmath.isMinusInfinity(x)){
                                        if(Fmath.isMinusInfinity(y))test=true;
                                }
                                else{
                                        if(x==y)test=true;
                                }
                        }
                }
                return test;
        }

        // Returns true if x equals y
        // x and y are int
        public static boolean isEqual(int x, int y){
                boolean test=false;
                if(x==y)test=true;
                return test;
        }

        // Returns true if x equals y
        // x and y are char
        public static boolean isEqual(char x, char y){
                boolean test=false;
                if(x==y)test=true;
                return test;
        }

        // Returns true if x equals y
        // x and y are Strings
        public static boolean isEqual(String x, String y){
                boolean test=false;
                if(x.equals(y))test=true;
                return test;
        }

        // Returns true if x is an even number, false if x is an odd number
        // x is int
        public static boolean isEven(int x){
                boolean test=false;
                if(x%2 == 0.0D)test=true;
                return test;
        }

        // Returns true if x is an even number, false if x is an odd number
        // x is float but must hold an integer value
        public static boolean isEven(float x){
                double y=Math.floor(x);
                if(((double)x - y)!= 0.0D)throw new IllegalArgumentException("the argument is not an integer");
                boolean test=false;
                y=Math.floor(x/2.0F);
                if(((double)(x/2.0F)-y) == 0.0D)test=true;
                return test;
        }

        // Returns true if x is an even number, false if x is an odd number
        // x is double but must hold an integer value
        public static boolean isEven(double x){
                double y=Math.floor(x);
                if((x - y)!= 0.0D)throw new IllegalArgumentException("the argument is not an integer");
                boolean test=false;
                y=Math.floor(x/2.0F);
                if((x/2.0D-y) == 0.0D)test=true;
                return test;
        }

        // Returns true if x is an odd number, false if x is an even number
        // x is int
        public static boolean isOdd(int x){
                boolean test=true;
                if(x%2 == 0.0D)test=false;
                return test;
        }

        // Returns true if x is an odd number, false if x is an even number
        // x is float but must hold an integer value
        public static boolean isOdd(float x){
                double y=Math.floor(x);
                if(((double)x - y)!= 0.0D)throw new IllegalArgumentException("the argument is not an integer");
                boolean test=true;
                y=Math.floor(x/2.0F);
                if(((double)(x/2.0F)-y) == 0.0D)test=false;
                return test;
        }

        // Returns true if x is an odd number, false if x is an even number
        // x is double but must hold an integer value
        public static boolean isOdd(double x){
                double y=Math.floor(x);
                if((x - y)!= 0.0D)throw new IllegalArgumentException("the argument is not an integer");
                boolean test=true;
                y=Math.floor(x/2.0F);
                if((x/2.0D-y) == 0.0D)test=false;
                return test;
        }
}





